c
c
c     ##########################################################
c     ##  COPYRIGHT (C) 2023 by Rae Corrigan & Jay W. Ponder  ##
c     ##                  All Rights Reserved                 ##
c     ##########################################################
c
c     #################################################################
c     ##                                                             ##
c     ##  subroutine initneck  --  set implicit solvent neck values  ##
c     ##                                                             ##
c     #################################################################
c
c
c     "initneck" sets the atom radius ranges as well as the neck
c     correction values for interstitial regions between atom pairs,
c     as used in implicit solvation models
c
c     literature reference:
c
c     R. A. Corrigan, A. C. Thiel, J. R. Lynn, T. L. Cassavant,
c     P. Ren, J. W. Ponder and M. J. Schnieders, "A Generalized
c     Kirkwood Implicit Solvent for the Polarizable AMOEBA Protein
c     Model", Journal of Chemical Physics, 159, 054102 (2023)
c
c
      subroutine initneck
      use solute
      implicit none
      integer i,j
      real*8 radbin(maxneck)
      real*8 aij(maxneck,maxneck)
      real*8 bij(maxneck,maxneck)
c
c
c     atom radius size bins (Angstroms)
c
      data radbin  / 0.80d0, 0.85d0, 0.90d0, 0.95d0, 1.00d0, 1.05d0,
     &               1.10d0, 1.15d0, 1.20d0, 1.25d0, 1.30d0, 1.35d0,
     &               1.40d0, 1.45d0, 1.50d0, 1.55d0, 1.60d0, 1.65d0,
     &               1.70d0, 1.75d0, 1.80d0, 1.85d0, 1.90d0, 1.95d0,
     &               2.00d0, 2.05d0, 2.10d0, 2.15d0, 2.20d0, 2.25d0,
     &               2.30d0, 2.35d0, 2.40d0, 2.45d0, 2.50d0, 2.55d0,
     &               2.60d0, 2.65d0, 2.70d0, 2.75d0, 2.80d0, 2.85d0,
     &               2.90d0, 2.95d0, 3.00d0 /
c
c     Aij neck correction constants
c
      data aij(1,:)  / 0.0000577616d0, 0.0000584661d0, 0.0000363925d0,
     &                 0.0000395472d0, 0.0000443202d0, 0.0000485507d0,
     &                 0.0000430862d0, 0.0000485067d0, 0.0000244504d0,
     &                 0.0000278293d0, 0.0000329908d0, 0.0000292135d0,
     &                 0.0000343621d0, 0.0000393724d0, 0.0000352501d0,
     &                 0.0000303823d0, 0.0000360595d0, 0.0000418690d0,
     &                 0.0000365804d0, 0.0000248824d0, 0.0000375656d0,
     &                 0.0000428918d0, 0.0000377450d0, 0.0000447160d0,
     &                 0.0000395375d0, 0.0000345934d0, 0.0000536114d0,
     &                 0.0000470958d0, 0.0000542111d0, 0.0000360263d0,
     &                 0.0000553398d0, 0.0000483197d0, 0.0000555466d0,
     &                 0.0000374290d0, 0.0000327412d0, 0.0000386635d0,
     &                 0.0000578345d0, 0.0000513194d0, 0.0000581077d0,
     &                 0.0000394022d0, 0.0000599546d0, 0.0000392112d0,
     &                 0.0000597613d0, 0.0000403186d0, 0.0000615149d0 /
      data aij(2,:)  / 0.0000446374d0, 0.0000460309d0, 0.0000475714d0,
     &                 0.0000225974d0, 0.0000318052d0, 0.0000275537d0,
     &                 0.0000296576d0, 0.0000339157d0, 0.0000295595d0,
     &                 0.0000344661d0, 0.0000395587d0, 0.0000197502d0,
     &                 0.0000300923d0, 0.0000272524d0, 0.0000187561d0,
     &                 0.0000272680d0, 0.0000247659d0, 0.0000214933d0,
     &                 0.0000252850d0, 0.0000223386d0, 0.0000261633d0,
     &                 0.0000398363d0, 0.0000363492d0, 0.0000412599d0,
     &                 0.0000363327d0, 0.0000422768d0, 0.0000367723d0,
     &                 0.0000425282d0, 0.0000374607d0, 0.0000325975d0,
     &                 0.0000383563d0, 0.0000253707d0, 0.0000301193d0,
     &                 0.0000200866d0, 0.0000299808d0, 0.0000263514d0,
     &                 0.0000304240d0, 0.0000349497d0, 0.0000408533d0,
     &                 0.0000268003d0, 0.0000407538d0, 0.0000357651d0,
     &                 0.0000418473d0, 0.0000278226d0, 0.0000419066d0 /
      data aij(3,:)  / 0.0000360721d0, 0.0000365794d0, 0.0000372415d0,
     &                 0.0000303637d0, 0.0000319835d0, 0.0000267215d0,
     &                 0.0000171087d0, 0.0000252702d0, 0.0000221094d0,
     &                 0.0000187747d0, 0.0000212341d0, 0.0000251617d0,
     &                 0.0000215761d0, 0.0000191079d0, 0.0000291389d0,
     &                 0.0000199177d0, 0.0000288958d0, 0.0000260343d0,
     &                 0.0000179669d0, 0.0000210581d0, 0.0000178820d0,
     &                 0.0000215615d0, 0.0000243911d0, 0.0000220007d0,
     &                 0.0000249689d0, 0.0000295075d0, 0.0000260062d0,
     &                 0.0000297050d0, 0.0000203093d0, 0.0000305802d0,
     &                 0.0000202272d0, 0.0000308984d0, 0.0000207930d0,
     &                 0.0000316764d0, 0.0000277135d0, 0.0000316983d0,
     &                 0.0000219362d0, 0.0000326865d0, 0.0000286724d0,
     &                 0.0000332929d0, 0.0000220858d0, 0.0000332871d0,
     &                 0.0000294921d0, 0.0000348374d0, 0.0000230989d0 /
      data aij(4,:)  / 0.0000398429d0, 0.0000223074d0, 0.0000298279d0,
     &                 0.0000315221d0, 0.0000187855d0, 0.0000264807d0,
     &                 0.0000169459d0, 0.0000180824d0, 0.0000162496d0,
     &                 0.0000233148d0, 0.0000157394d0, 0.0000174931d0,
     &                 0.0000152866d0, 0.0000230620d0, 0.0000205539d0,
     &                 0.0000234872d0, 0.0000213273d0, 0.0000181123d0,
     &                 0.0000210938d0, 0.0000186307d0, 0.0000216437d0,
     &                 0.0000146084d0, 0.0000222850d0, 0.0000149429d0,
     &                 0.0000170964d0, 0.0000153536d0, 0.0000177037d0,
     &                 0.0000208958d0, 0.0000237412d0, 0.0000164214d0,
     &                 0.0000239561d0, 0.0000166849d0, 0.0000188702d0,
     &                 0.0000167221d0, 0.0000188448d0, 0.0000172644d0,
     &                 0.0000194706d0, 0.0000174585d0, 0.0000259642d0,
     &                 0.0000177409d0, 0.0000199219d0, 0.0000235990d0,
     &                 0.0000202504d0, 0.0000234495d0, 0.0000156052d0 /
      data aij(5,:)  / 0.0000435754d0, 0.0000245527d0, 0.0000318302d0,
     &                 0.0000188461d0, 0.0000192238d0, 0.0000208396d0,
     &                 0.0000169209d0, 0.0000183816d0, 0.0000204363d0,
     &                 0.0000131412d0, 0.0000193141d0, 0.0000097655d0,
     &                 0.0000110841d0, 0.0000129627d0, 0.0000112003d0,
     &                 0.0000130467d0, 0.0000113669d0, 0.0000168195d0,
     &                 0.0000117549d0, 0.0000134876d0, 0.0000118881d0,
     &                 0.0000180528d0, 0.0000159126d0, 0.0000181928d0,
     &                 0.0000125940d0, 0.0000183351d0, 0.0000125678d0,
     &                 0.0000145035d0, 0.0000131847d0, 0.0000144906d0,
     &                 0.0000129840d0, 0.0000150912d0, 0.0000176485d0,
     &                 0.0000153063d0, 0.0000178114d0, 0.0000157934d0,
     &                 0.0000180763d0, 0.0000157725d0, 0.0000186835d0,
     &                 0.0000121847d0, 0.0000185069d0, 0.0000161900d0,
     &                 0.0000191223d0, 0.0000126643d0, 0.0000192108d0 /
      data aij(6,:)  / 0.0000505580d0, 0.0000264633d0, 0.0000205166d0,
     &                 0.0000195164d0, 0.0000204987d0, 0.0000160556d0,
     &                 0.0000127648d0, 0.0000137952d0, 0.0000155458d0,
     &                 0.0000176400d0, 0.0000146241d0, 0.0000124027d0,
     &                 0.0000103247d0, 0.0000120745d0, 0.0000105222d0,
     &                 0.0000124882d0, 0.0000140152d0, 0.0000095449d0,
     &                 0.0000141168d0, 0.0000074383d0, 0.0000142994d0,
     &                 0.0000098671d0, 0.0000112045d0, 0.0000132832d0,
     &                 0.0000113876d0, 0.0000135139d0, 0.0000116918d0,
     &                 0.0000137152d0, 0.0000118068d0, 0.0000140495d0,
     &                 0.0000093075d0, 0.0000144645d0, 0.0000126001d0,
     &                 0.0000142255d0, 0.0000128571d0, 0.0000148412d0,
     &                 0.0000128819d0, 0.0000152166d0, 0.0000099735d0,
     &                 0.0000151882d0, 0.0000133944d0, 0.0000154829d0,
     &                 0.0000135105d0, 0.0000154771d0, 0.0000106154d0 /
      data aij(7,:)  / 0.0000449657d0, 0.0000230612d0, 0.0000221541d0,
     &                 0.0000166532d0, 0.0000171555d0, 0.0000227174d0,
     &                 0.0000182899d0, 0.0000193198d0, 0.0000118768d0,
     &                 0.0000130396d0, 0.0000113361d0, 0.0000093799d0,
     &                 0.0000108573d0, 0.0000120843d0, 0.0000106342d0,
     &                 0.0000117938d0, 0.0000080569d0, 0.0000091361d0,
     &                 0.0000107718d0, 0.0000123647d0, 0.0000106453d0,
     &                 0.0000123678d0, 0.0000109954d0, 0.0000126151d0,
     &                 0.0000145692d0, 0.0000128038d0, 0.0000146248d0,
     &                 0.0000099091d0, 0.0000114200d0, 0.0000059787d0,
     &                 0.0000117991d0, 0.0000136370d0, 0.0000119830d0,
     &                 0.0000080880d0, 0.0000120147d0, 0.0000108970d0,
     &                 0.0000161522d0, 0.0000083711d0, 0.0000094980d0,
     &                 0.0000112647d0, 0.0000096319d0, 0.0000113677d0,
     &                 0.0000127823d0, 0.0000114549d0, 0.0000100471d0 /
      data aij(8,:)  / 0.0000280510d0, 0.0000266023d0, 0.0000144227d0,
     &                 0.0000188499d0, 0.0000184712d0, 0.0000146008d0,
     &                 0.0000195724d0, 0.0000161268d0, 0.0000073492d0,
     &                 0.0000137704d0, 0.0000111866d0, 0.0000072795d0,
     &                 0.0000081037d0, 0.0000092319d0, 0.0000078144d0,
     &                 0.0000089354d0, 0.0000102556d0, 0.0000090005d0,
     &                 0.0000059425d0, 0.0000090194d0, 0.0000080186d0,
     &                 0.0000069127d0, 0.0000082105d0, 0.0000093797d0,
     &                 0.0000082401d0, 0.0000093675d0, 0.0000082671d0,
     &                 0.0000071727d0, 0.0000064889d0, 0.0000097376d0,
     &                 0.0000087729d0, 0.0000098019d0, 0.0000088064d0,
     &                 0.0000059057d0, 0.0000089896d0, 0.0000077635d0,
     &                 0.0000070336d0, 0.0000047870d0, 0.0000094273d0,
     &                 0.0000080466d0, 0.0000093429d0, 0.0000083051d0,
     &                 0.0000096725d0, 0.0000109657d0, 0.0000097148d0 /
      data aij(9,:)  / 0.0000325317d0, 0.0000295175d0, 0.0000213868d0,
     &                 0.0000208878d0, 0.0000153087d0, 0.0000116695d0,
     &                 0.0000091353d0, 0.0000128753d0, 0.0000102693d0,
     &                 0.0000107086d0, 0.0000088500d0, 0.0000098955d0,
     &                 0.0000081413d0, 0.0000052255d0, 0.0000078083d0,
     &                 0.0000067997d0, 0.0000098737d0, 0.0000066979d0,
     &                 0.0000098507d0, 0.0000051252d0, 0.0000074307d0,
     &                 0.0000067922d0, 0.0000057491d0, 0.0000068223d0,
     &                 0.0000058920d0, 0.0000070420d0, 0.0000060742d0,
     &                 0.0000069635d0, 0.0000062687d0, 0.0000054473d0,
     &                 0.0000061911d0, 0.0000073394d0, 0.0000037836d0,
     &                 0.0000074661d0, 0.0000084983d0, 0.0000073838d0,
     &                 0.0000087027d0, 0.0000076952d0, 0.0000088636d0,
     &                 0.0000102208d0, 0.0000090108d0, 0.0000104452d0,
     &                 0.0000090944d0, 0.0000104765d0, 0.0000070232d0 /
      data aij(10,:) / 0.0000289111d0, 0.0000198943d0, 0.0000241993d0,
     &                 0.0000238536d0, 0.0000182705d0, 0.0000129878d0,
     &                 0.0000076255d0, 0.0000102719d0, 0.0000112418d0,
     &                 0.0000067659d0, 0.0000071756d0, 0.0000100155d0,
     &                 0.0000065300d0, 0.0000071811d0, 0.0000060818d0,
     &                 0.0000066768d0, 0.0000059369d0, 0.0000065993d0,
     &                 0.0000076971d0, 0.0000063984d0, 0.0000057069d0,
     &                 0.0000086185d0, 0.0000076581d0, 0.0000051018d0,
     &                 0.0000057661d0, 0.0000051292d0, 0.0000058121d0,
     &                 0.0000050955d0, 0.0000079236d0, 0.0000069082d0,
     &                 0.0000060582d0, 0.0000041682d0, 0.0000080245d0,
     &                 0.0000042302d0, 0.0000082046d0, 0.0000056029d0,
     &                 0.0000048280d0, 0.0000057624d0, 0.0000065014d0,
     &                 0.0000058246d0, 0.0000049969d0, 0.0000058491d0,
     &                 0.0000066001d0, 0.0000059999d0, 0.0000088507d0 /
      data aij(11,:) / 0.0000265006d0, 0.0000302279d0, 0.0000215102d0,
     &                 0.0000203741d0, 0.0000195197d0, 0.0000114271d0,
     &                 0.0000110442d0, 0.0000113577d0, 0.0000067378d0,
     &                 0.0000072687d0, 0.0000074058d0, 0.0000047733d0,
     &                 0.0000067184d0, 0.0000076291d0, 0.0000062951d0,
     &                 0.0000051957d0, 0.0000044871d0, 0.0000050768d0,
     &                 0.0000044211d0, 0.0000049880d0, 0.0000043518d0,
     &                 0.0000048796d0, 0.0000057445d0, 0.0000049576d0,
     &                 0.0000057224d0, 0.0000037798d0, 0.0000043883d0,
     &                 0.0000066161d0, 0.0000044279d0, 0.0000050580d0,
     &                 0.0000045758d0, 0.0000051723d0, 0.0000046936d0,
     &                 0.0000052703d0, 0.0000047182d0, 0.0000041352d0,
     &                 0.0000063088d0, 0.0000041672d0, 0.0000063473d0,
     &                 0.0000055519d0, 0.0000049210d0, 0.0000033576d0,
     &                 0.0000049766d0, 0.0000033829d0, 0.0000051220d0 /
      data aij(12,:) / 0.0000393068d0, 0.0000205177d0, 0.0000254533d0,
     &                 0.0000229604d0, 0.0000129838d0, 0.0000126427d0,
     &                 0.0000096583d0, 0.0000123670d0, 0.0000100764d0,
     &                 0.0000100625d0, 0.0000047514d0, 0.0000049466d0,
     &                 0.0000041283d0, 0.0000059449d0, 0.0000065208d0,
     &                 0.0000072136d0, 0.0000045347d0, 0.0000039272d0,
     &                 0.0000033654d0, 0.0000038633d0, 0.0000032913d0,
     &                 0.0000038211d0, 0.0000041976d0, 0.0000037860d0,
     &                 0.0000042838d0, 0.0000037697d0, 0.0000043641d0,
     &                 0.0000038556d0, 0.0000033089d0, 0.0000039242d0,
     &                 0.0000043859d0, 0.0000051678d0, 0.0000034265d0,
     &                 0.0000040012d0, 0.0000035224d0, 0.0000053110d0,
     &                 0.0000046531d0, 0.0000053869d0, 0.0000036188d0,
     &                 0.0000054308d0, 0.0000048054d0, 0.0000055242d0,
     &                 0.0000048512d0, 0.0000056159d0, 0.0000037853d0 /
      data aij(13,:) / 0.0000340677d0, 0.0000307596d0, 0.0000223668d0,
     &                 0.0000152946d0, 0.0000146712d0, 0.0000106255d0,
     &                 0.0000106032d0, 0.0000081832d0, 0.0000086244d0,
     &                 0.0000049411d0, 0.0000067545d0, 0.0000040421d0,
     &                 0.0000044178d0, 0.0000036241d0, 0.0000052208d0,
     &                 0.0000032905d0, 0.0000048780d0, 0.0000053199d0,
     &                 0.0000046459d0, 0.0000039440d0, 0.0000026124d0,
     &                 0.0000050097d0, 0.0000026014d0, 0.0000029505d0,
     &                 0.0000032710d0, 0.0000037980d0, 0.0000043410d0,
     &                 0.0000037740d0, 0.0000033662d0, 0.0000050175d0,
     &                 0.0000033958d0, 0.0000051015d0, 0.0000034086d0,
     &                 0.0000030335d0, 0.0000034235d0, 0.0000039828d0,
     &                 0.0000045344d0, 0.0000031386d0, 0.0000046458d0,
     &                 0.0000041624d0, 0.0000036086d0, 0.0000041947d0,
     &                 0.0000028209d0, 0.0000033460d0, 0.0000028613d0 /
      data aij(14,:) / 0.0000305973d0, 0.0000272968d0, 0.0000142276d0,
     &                 0.0000179127d0, 0.0000097246d0, 0.0000127507d0,
     &                 0.0000070473d0, 0.0000090593d0, 0.0000092161d0,
     &                 0.0000074076d0, 0.0000074685d0, 0.0000058617d0,
     &                 0.0000047016d0, 0.0000051588d0, 0.0000056952d0,
     &                 0.0000035730d0, 0.0000038387d0, 0.0000042402d0,
     &                 0.0000035579d0, 0.0000018248d0, 0.0000046133d0,
     &                 0.0000039370d0, 0.0000045159d0, 0.0000029244d0,
     &                 0.0000019710d0, 0.0000022822d0, 0.0000025439d0,
     &                 0.0000028944d0, 0.0000025756d0, 0.0000029250d0,
     &                 0.0000034260d0, 0.0000050361d0, 0.0000034605d0,
     &                 0.0000038688d0, 0.0000026949d0, 0.0000014049d0,
     &                 0.0000027321d0, 0.0000023567d0, 0.0000027913d0,
     &                 0.0000031218d0, 0.0000027763d0, 0.0000031969d0,
     &                 0.0000037163d0, 0.0000032316d0, 0.0000037723d0 /
      data aij(15,:) / 0.0000363524d0, 0.0000185238d0, 0.0000223944d0,
     &                 0.0000156641d0, 0.0000146928d0, 0.0000107770d0,
     &                 0.0000105276d0, 0.0000102180d0, 0.0000079337d0,
     &                 0.0000078580d0, 0.0000062276d0, 0.0000066384d0,
     &                 0.0000039738d0, 0.0000055727d0, 0.0000060414d0,
     &                 0.0000037461d0, 0.0000023444d0, 0.0000033734d0,
     &                 0.0000028739d0, 0.0000032262d0, 0.0000026725d0,
     &                 0.0000030603d0, 0.0000025932d0, 0.0000030096d0,
     &                 0.0000019791d0, 0.0000029386d0, 0.0000033334d0,
     &                 0.0000029324d0, 0.0000025033d0, 0.0000017645d0,
     &                 0.0000014959d0, 0.0000022899d0, 0.0000025847d0,
     &                 0.0000030273d0, 0.0000026349d0, 0.0000039712d0,
     &                 0.0000026279d0, 0.0000040671d0, 0.0000020776d0,
     &                 0.0000023832d0, 0.0000020939d0, 0.0000024085d0,
     &                 0.0000021149d0, 0.0000041807d0, 0.0000016785d0 /
      data aij(16,:) / 0.0000531068d0, 0.0000206882d0, 0.0000259899d0,
     &                 0.0000178808d0, 0.0000168038d0, 0.0000128055d0,
     &                 0.0000119227d0, 0.0000069103d0, 0.0000090425d0,
     &                 0.0000068286d0, 0.0000069793d0, 0.0000031585d0,
     &                 0.0000042989d0, 0.0000027028d0, 0.0000036738d0,
     &                 0.0000031193d0, 0.0000043434d0, 0.0000035227d0,
     &                 0.0000030391d0, 0.0000024934d0, 0.0000027944d0,
     &                 0.0000024067d0, 0.0000027907d0, 0.0000023644d0,
     &                 0.0000026945d0, 0.0000022659d0, 0.0000020013d0,
     &                 0.0000029627d0, 0.0000026172d0, 0.0000017442d0,
     &                 0.0000025980d0, 0.0000017461d0, 0.0000020037d0,
     &                 0.0000017818d0, 0.0000026313d0, 0.0000023219d0,
     &                 0.0000026522d0, 0.0000023901d0, 0.0000026985d0,
     &                 0.0000014157d0, 0.0000027079d0, 0.0000018522d0,
     &                 0.0000016312d0, 0.0000019179d0, 0.0000016703d0 /
      data aij(17,:) / 0.0000471466d0, 0.0000321678d0, 0.0000221166d0,
     &                 0.0000210526d0, 0.0000114910d0, 0.0000108287d0,
     &                 0.0000080777d0, 0.0000104203d0, 0.0000101134d0,
     &                 0.0000058443d0, 0.0000045067d0, 0.0000045975d0,
     &                 0.0000027454d0, 0.0000037782d0, 0.0000039918d0,
     &                 0.0000043093d0, 0.0000046405d0, 0.0000029403d0,
     &                 0.0000023896d0, 0.0000020792d0, 0.0000029414d0,
     &                 0.0000014845d0, 0.0000016390d0, 0.0000018155d0,
     &                 0.0000020600d0, 0.0000018173d0, 0.0000020305d0,
     &                 0.0000018045d0, 0.0000020275d0, 0.0000023056d0,
     &                 0.0000026380d0, 0.0000022906d0, 0.0000015633d0,
     &                 0.0000022861d0, 0.0000015702d0, 0.0000022925d0,
     &                 0.0000015926d0, 0.0000017741d0, 0.0000015991d0,
     &                 0.0000017969d0, 0.0000021445d0, 0.0000018491d0,
     &                 0.0000021663d0, 0.0000018937d0, 0.0000021829d0 /
      data aij(18,:) / 0.0000415033d0, 0.0000375035d0, 0.0000257317d0,
     &                 0.0000181483d0, 0.0000170401d0, 0.0000165719d0,
     &                 0.0000091912d0, 0.0000089667d0, 0.0000066295d0,
     &                 0.0000051660d0, 0.0000051868d0, 0.0000051552d0,
     &                 0.0000041581d0, 0.0000041949d0, 0.0000044142d0,
     &                 0.0000035882d0, 0.0000029449d0, 0.0000031798d0,
     &                 0.0000020273d0, 0.0000021690d0, 0.0000024377d0,
     &                 0.0000026775d0, 0.0000029680d0, 0.0000014909d0,
     &                 0.0000021480d0, 0.0000018690d0, 0.0000027312d0,
     &                 0.0000014150d0, 0.0000020266d0, 0.0000013942d0,
     &                 0.0000015678d0, 0.0000014176d0, 0.0000015601d0,
     &                 0.0000018280d0, 0.0000020424d0, 0.0000023951d0,
     &                 0.0000015782d0, 0.0000024117d0, 0.0000016197d0,
     &                 0.0000014155d0, 0.0000012558d0, 0.0000018686d0,
     &                 0.0000016359d0, 0.0000018949d0, 0.0000010017d0 /
      data aij(19,:) / 0.0000468296d0, 0.0000330068d0, 0.0000237247d0,
     &                 0.0000165336d0, 0.0000150520d0, 0.0000109676d0,
     &                 0.0000103584d0, 0.0000059806d0, 0.0000075019d0,
     &                 0.0000075639d0, 0.0000075577d0, 0.0000044171d0,
     &                 0.0000035406d0, 0.0000027896d0, 0.0000038483d0,
     &                 0.0000030165d0, 0.0000024502d0, 0.0000020229d0,
     &                 0.0000037725d0, 0.0000023248d0, 0.0000025725d0,
     &                 0.0000016526d0, 0.0000018416d0, 0.0000020307d0,
     &                 0.0000023292d0, 0.0000019462d0, 0.0000022344d0,
     &                 0.0000014452d0, 0.0000016743d0, 0.0000024751d0,
     &                 0.0000021503d0, 0.0000010914d0, 0.0000016308d0,
     &                 0.0000018570d0, 0.0000021221d0, 0.0000010982d0,
     &                 0.0000016425d0, 0.0000011133d0, 0.0000021459d0,
     &                 0.0000018960d0, 0.0000021557d0, 0.0000011360d0,
     &                 0.0000012755d0, 0.0000011503d0, 0.0000012923d0 /
      data aij(20,:) / 0.0000326803d0, 0.0000295857d0, 0.0000154475d0,
     &                 0.0000195504d0, 0.0000133752d0, 0.0000130826d0,
     &                 0.0000093108d0, 0.0000090846d0, 0.0000051384d0,
     &                 0.0000065258d0, 0.0000049655d0, 0.0000050318d0,
     &                 0.0000039301d0, 0.0000031064d0, 0.0000018937d0,
     &                 0.0000033983d0, 0.0000027192d0, 0.0000021941d0,
     &                 0.0000023262d0, 0.0000011684d0, 0.0000015864d0,
     &                 0.0000017653d0, 0.0000019394d0, 0.0000021477d0,
     &                 0.0000018003d0, 0.0000015654d0, 0.0000017271d0,
     &                 0.0000019541d0, 0.0000012911d0, 0.0000008739d0,
     &                 0.0000016831d0, 0.0000011063d0, 0.0000016687d0,
     &                 0.0000014377d0, 0.0000016576d0, 0.0000010945d0,
     &                 0.0000012820d0, 0.0000010911d0, 0.0000007782d0,
     &                 0.0000014342d0, 0.0000013165d0, 0.0000018913d0,
     &                 0.0000013170d0, 0.0000008899d0, 0.0000013269d0 /
      data aij(21,:) / 0.0000381583d0, 0.0000264565d0, 0.0000136612d0,
     &                 0.0000212466d0, 0.0000153348d0, 0.0000145316d0,
     &                 0.0000138739d0, 0.0000060648d0, 0.0000076347d0,
     &                 0.0000074893d0, 0.0000043282d0, 0.0000057183d0,
     &                 0.0000033840d0, 0.0000045556d0, 0.0000035441d0,
     &                 0.0000028461d0, 0.0000022853d0, 0.0000031916d0,
     &                 0.0000026005d0, 0.0000021084d0, 0.0000013651d0,
     &                 0.0000024956d0, 0.0000016110d0, 0.0000017750d0,
     &                 0.0000014880d0, 0.0000016813d0, 0.0000010862d0,
     &                 0.0000016184d0, 0.0000013616d0, 0.0000015542d0,
     &                 0.0000017206d0, 0.0000015449d0, 0.0000013086d0,
     &                 0.0000015137d0, 0.0000012941d0, 0.0000011585d0,
     &                 0.0000012793d0, 0.0000008977d0, 0.0000009980d0,
     &                 0.0000011586d0, 0.0000010254d0, 0.0000011655d0,
     &                 0.0000010218d0, 0.0000011671d0, 0.0000010320d0 /
      data aij(22,:) / 0.0000435206d0, 0.0000394602d0, 0.0000280329d0,
     &                 0.0000143528d0, 0.0000103966d0, 0.0000099195d0,
     &                 0.0000123591d0, 0.0000070629d0, 0.0000087274d0,
     &                 0.0000066090d0, 0.0000048718d0, 0.0000038172d0,
     &                 0.0000050202d0, 0.0000029810d0, 0.0000040378d0,
     &                 0.0000024193d0, 0.0000024874d0, 0.0000015771d0,
     &                 0.0000016403d0, 0.0000017939d0, 0.0000019398d0,
     &                 0.0000012193d0, 0.0000017244d0, 0.0000008475d0,
     &                 0.0000009524d0, 0.0000008041d0, 0.0000011807d0,
     &                 0.0000009651d0, 0.0000011501d0, 0.0000012202d0,
     &                 0.0000010892d0, 0.0000011977d0, 0.0000010599d0,
     &                 0.0000015547d0, 0.0000010348d0, 0.0000011765d0,
     &                 0.0000010241d0, 0.0000009188d0, 0.0000007943d0,
     &                 0.0000009237d0, 0.0000010314d0, 0.0000009055d0,
     &                 0.0000010352d0, 0.0000009196d0, 0.0000013384d0 /
      data aij(23,:) / 0.0000379248d0, 0.0000353934d0, 0.0000241508d0,
     &                 0.0000168891d0, 0.0000210529d0, 0.0000086678d0,
     &                 0.0000083510d0, 0.0000106765d0, 0.0000058348d0,
     &                 0.0000057662d0, 0.0000074621d0, 0.0000043406d0,
     &                 0.0000033715d0, 0.0000044195d0, 0.0000020552d0,
     &                 0.0000027516d0, 0.0000016638d0, 0.0000022213d0,
     &                 0.0000023581d0, 0.0000019527d0, 0.0000016003d0,
     &                 0.0000017368d0, 0.0000008516d0, 0.0000009200d0,
     &                 0.0000016828d0, 0.0000008623d0, 0.0000009286d0,
     &                 0.0000006197d0, 0.0000011498d0, 0.0000007848d0,
     &                 0.0000006628d0, 0.0000009788d0, 0.0000010822d0,
     &                 0.0000012385d0, 0.0000008152d0, 0.0000009435d0,
     &                 0.0000010686d0, 0.0000012014d0, 0.0000006276d0,
     &                 0.0000007030d0, 0.0000010605d0, 0.0000012119d0,
     &                 0.0000006389d0, 0.0000007195d0, 0.0000006456d0 /
      data aij(24,:) / 0.0000457913d0, 0.0000308903d0, 0.0000282516d0,
     &                 0.0000197423d0, 0.0000139036d0, 0.0000076554d0,
     &                 0.0000101164d0, 0.0000092286d0, 0.0000068324d0,
     &                 0.0000052054d0, 0.0000050091d0, 0.0000049163d0,
     &                 0.0000039143d0, 0.0000029462d0, 0.0000039499d0,
     &                 0.0000031099d0, 0.0000024405d0, 0.0000014864d0,
     &                 0.0000012048d0, 0.0000021759d0, 0.0000013931d0,
     &                 0.0000014556d0, 0.0000009208d0, 0.0000017108d0,
     &                 0.0000011004d0, 0.0000011923d0, 0.0000007721d0,
     &                 0.0000006708d0, 0.0000005731d0, 0.0000008092d0,
     &                 0.0000009072d0, 0.0000007914d0, 0.0000008763d0,
     &                 0.0000007716d0, 0.0000006536d0, 0.0000007653d0,
     &                 0.0000008561d0, 0.0000009956d0, 0.0000008472d0,
     &                 0.0000007539d0, 0.0000008319d0, 0.0000005765d0,
     &                 0.0000008451d0, 0.0000009713d0, 0.0000008480d0 /
      data aij(25,:) / 0.0000514751d0, 0.0000274557d0, 0.0000241481d0,
     &                 0.0000174373d0, 0.0000161298d0, 0.0000116871d0,
     &                 0.0000084217d0, 0.0000081004d0, 0.0000058484d0,
     &                 0.0000056938d0, 0.0000057532d0, 0.0000043019d0,
     &                 0.0000032347d0, 0.0000025748d0, 0.0000034234d0,
     &                 0.0000026896d0, 0.0000016029d0, 0.0000016452d0,
     &                 0.0000023736d0, 0.0000013932d0, 0.0000011273d0,
     &                 0.0000009500d0, 0.0000013052d0, 0.0000010750d0,
     &                 0.0000012044d0, 0.0000010077d0, 0.0000011046d0,
     &                 0.0000007011d0, 0.0000010268d0, 0.0000008714d0,
     &                 0.0000012822d0, 0.0000010702d0, 0.0000007283d0,
     &                 0.0000010366d0, 0.0000005479d0, 0.0000007812d0,
     &                 0.0000005332d0, 0.0000007670d0, 0.0000006899d0,
     &                 0.0000007700d0, 0.0000006826d0, 0.0000007637d0,
     &                 0.0000008807d0, 0.0000004626d0, 0.0000006769d0 /
      data aij(26,:) / 0.0000347279d0, 0.0000326038d0, 0.0000222389d0,
     &                 0.0000203032d0, 0.0000186389d0, 0.0000135396d0,
     &                 0.0000074878d0, 0.0000093982d0, 0.0000069447d0,
     &                 0.0000051305d0, 0.0000038953d0, 0.0000049720d0,
     &                 0.0000038543d0, 0.0000038303d0, 0.0000017387d0,
     &                 0.0000013908d0, 0.0000018492d0, 0.0000018676d0,
     &                 0.0000011554d0, 0.0000016034d0, 0.0000009913d0,
     &                 0.0000010369d0, 0.0000011030d0, 0.0000015670d0,
     &                 0.0000013066d0, 0.0000011031d0, 0.0000009195d0,
     &                 0.0000007778d0, 0.0000010981d0, 0.0000009562d0,
     &                 0.0000010346d0, 0.0000011776d0, 0.0000007583d0,
     &                 0.0000005111d0, 0.0000005726d0, 0.0000011004d0,
     &                 0.0000009317d0, 0.0000004887d0, 0.0000005500d0,
     &                 0.0000008051d0, 0.0000005530d0, 0.0000004756d0,
     &                 0.0000005513d0, 0.0000006031d0, 0.0000005515d0 /
      data aij(27,:) / 0.0000404627d0, 0.0000279374d0, 0.0000254737d0,
     &                 0.0000173416d0, 0.0000166276d0, 0.0000090743d0,
     &                 0.0000084501d0, 0.0000108659d0, 0.0000080225d0,
     &                 0.0000045088d0, 0.0000044169d0, 0.0000057245d0,
     &                 0.0000033186d0, 0.0000043660d0, 0.0000034171d0,
     &                 0.0000015607d0, 0.0000020522d0, 0.0000012726d0,
     &                 0.0000016999d0, 0.0000013417d0, 0.0000018688d0,
     &                 0.0000015245d0, 0.0000009452d0, 0.0000007794d0,
     &                 0.0000008405d0, 0.0000011825d0, 0.0000007598d0,
     &                 0.0000008210d0, 0.0000007108d0, 0.0000007647d0,
     &                 0.0000006612d0, 0.0000007339d0, 0.0000008145d0,
     &                 0.0000007040d0, 0.0000006095d0, 0.0000008740d0,
     &                 0.0000003513d0, 0.0000006717d0, 0.0000005660d0,
     &                 0.0000004997d0, 0.0000009522d0, 0.0000005060d0,
     &                 0.0000005619d0, 0.0000005063d0, 0.0000004323d0 /
      data aij(28,:) / 0.0000472063d0, 0.0000323179d0, 0.0000294752d0,
     &                 0.0000207416d0, 0.0000113045d0, 0.0000136112d0,
     &                 0.0000129076d0, 0.0000071735d0, 0.0000091135d0,
     &                 0.0000052733d0, 0.0000039191d0, 0.0000038507d0,
     &                 0.0000022739d0, 0.0000029313d0, 0.0000017476d0,
     &                 0.0000029817d0, 0.0000017682d0, 0.0000014044d0,
     &                 0.0000018903d0, 0.0000011418d0, 0.0000015945d0,
     &                 0.0000009870d0, 0.0000010526d0, 0.0000011160d0,
     &                 0.0000011997d0, 0.0000013110d0, 0.0000010830d0,
     &                 0.0000006980d0, 0.0000009891d0, 0.0000006386d0,
     &                 0.0000005495d0, 0.0000006040d0, 0.0000006782d0,
     &                 0.0000004347d0, 0.0000004971d0, 0.0000004232d0,
     &                 0.0000004838d0, 0.0000006935d0, 0.0000003617d0,
     &                 0.0000004015d0, 0.0000003611d0, 0.0000003155d0,
     &                 0.0000005902d0, 0.0000003968d0, 0.0000004545d0 /
      data aij(29,:) / 0.0000534708d0, 0.0000373744d0, 0.0000196796d0,
     &                 0.0000137544d0, 0.0000129458d0, 0.0000119235d0,
     &                 0.0000088305d0, 0.0000111632d0, 0.0000062367d0,
     &                 0.0000059625d0, 0.0000034417d0, 0.0000044561d0,
     &                 0.0000033301d0, 0.0000025232d0, 0.0000019586d0,
     &                 0.0000027071d0, 0.0000015804d0, 0.0000012263d0,
     &                 0.0000016555d0, 0.0000017219d0, 0.0000010458d0,
     &                 0.0000008561d0, 0.0000006975d0, 0.0000009600d0,
     &                 0.0000013346d0, 0.0000008501d0, 0.0000004210d0,
     &                 0.0000005776d0, 0.0000008267d0, 0.0000009169d0,
     &                 0.0000004551d0, 0.0000005061d0, 0.0000007113d0,
     &                 0.0000006128d0, 0.0000006790d0, 0.0000004531d0,
     &                 0.0000005155d0, 0.0000005740d0, 0.0000006449d0,
     &                 0.0000002634d0, 0.0000004837d0, 0.0000005547d0,
     &                 0.0000004790d0, 0.0000004165d0, 0.0000004781d0 /
      data aij(30,:) / 0.0000471509d0, 0.0000331027d0, 0.0000226179d0,
     &                 0.0000161311d0, 0.0000144893d0, 0.0000107687d0,
     &                 0.0000104009d0, 0.0000128476d0, 0.0000071831d0,
     &                 0.0000053734d0, 0.0000030440d0, 0.0000051224d0,
     &                 0.0000039650d0, 0.0000022452d0, 0.0000017674d0,
     &                 0.0000017430d0, 0.0000017813d0, 0.0000014173d0,
     &                 0.0000018887d0, 0.0000008757d0, 0.0000012001d0,
     &                 0.0000016279d0, 0.0000012946d0, 0.0000004971d0,
     &                 0.0000008642d0, 0.0000005591d0, 0.0000006041d0,
     &                 0.0000006461d0, 0.0000005270d0, 0.0000007557d0,
     &                 0.0000004882d0, 0.0000007062d0, 0.0000005940d0,
     &                 0.0000008734d0, 0.0000005615d0, 0.0000004848d0,
     &                 0.0000004202d0, 0.0000004705d0, 0.0000004021d0,
     &                 0.0000004565d0, 0.0000003960d0, 0.0000004488d0,
     &                 0.0000004998d0, 0.0000005792d0, 0.0000003821d0 /
      data aij(31,:) / 0.0000408712d0, 0.0000290825d0, 0.0000201809d0,
     &                 0.0000184693d0, 0.0000127543d0, 0.0000122856d0,
     &                 0.0000089579d0, 0.0000087349d0, 0.0000048255d0,
     &                 0.0000062188d0, 0.0000059289d0, 0.0000044984d0,
     &                 0.0000019907d0, 0.0000033642d0, 0.0000034197d0,
     &                 0.0000026227d0, 0.0000012039d0, 0.0000015836d0,
     &                 0.0000009945d0, 0.0000013051d0, 0.0000017849d0,
     &                 0.0000010646d0, 0.0000005240d0, 0.0000007062d0,
     &                 0.0000005767d0, 0.0000006087d0, 0.0000008726d0,
     &                 0.0000005467d0, 0.0000007758d0, 0.0000006348d0,
     &                 0.0000007039d0, 0.0000003514d0, 0.0000006567d0,
     &                 0.0000002671d0, 0.0000006217d0, 0.0000006907d0,
     &                 0.0000005847d0, 0.0000001470d0, 0.0000004367d0,
     &                 0.0000003746d0, 0.0000003283d0, 0.0000003639d0,
     &                 0.0000004146d0, 0.0000003630d0, 0.0000004069d0 /
      data aij(32,:) / 0.0000365867d0, 0.0000334092d0, 0.0000233935d0,
     &                 0.0000211266d0, 0.0000148841d0, 0.0000145388d0,
     &                 0.0000102065d0, 0.0000099718d0, 0.0000095120d0,
     &                 0.0000041586d0, 0.0000052071d0, 0.0000030423d0,
     &                 0.0000030232d0, 0.0000022968d0, 0.0000023000d0,
     &                 0.0000030274d0, 0.0000022943d0, 0.0000010722d0,
     &                 0.0000011058d0, 0.0000014789d0, 0.0000009014d0,
     &                 0.0000009248d0, 0.0000009830d0, 0.0000010354d0,
     &                 0.0000014149d0, 0.0000011942d0, 0.0000005606d0,
     &                 0.0000006027d0, 0.0000005065d0, 0.0000004217d0,
     &                 0.0000005883d0, 0.0000005132d0, 0.0000003302d0,
     &                 0.0000004589d0, 0.0000006648d0, 0.0000003397d0,
     &                 0.0000004773d0, 0.0000003190d0, 0.0000004559d0,
     &                 0.0000003103d0, 0.0000005849d0, 0.0000003016d0,
     &                 0.0000005635d0, 0.0000003806d0, 0.0000002617d0 /
      data aij(33,:) / 0.0000424740d0, 0.0000299616d0, 0.0000274655d0,
     &                 0.0000188374d0, 0.0000175698d0, 0.0000126091d0,
     &                 0.0000089948d0, 0.0000068101d0, 0.0000083814d0,
     &                 0.0000048968d0, 0.0000060247d0, 0.0000059532d0,
     &                 0.0000020417d0, 0.0000044737d0, 0.0000025919d0,
     &                 0.0000020128d0, 0.0000020369d0, 0.0000012489d0,
     &                 0.0000016173d0, 0.0000012975d0, 0.0000013224d0,
     &                 0.0000010382d0, 0.0000008390d0, 0.0000008790d0,
     &                 0.0000009452d0, 0.0000009921d0, 0.0000006307d0,
     &                 0.0000006689d0, 0.0000003338d0, 0.0000007868d0,
     &                 0.0000005076d0, 0.0000004219d0, 0.0000003654d0,
     &                 0.0000008632d0, 0.0000004278d0, 0.0000003612d0,
     &                 0.0000002445d0, 0.0000005864d0, 0.0000002315d0,
     &                 0.0000004357d0, 0.0000002247d0, 0.0000004156d0,
     &                 0.0000002193d0, 0.0000003178d0, 0.0000002730d0 /
      data aij(34,:) / 0.0000503599d0, 0.0000451538d0, 0.0000316354d0,
     &                 0.0000217532d0, 0.0000202497d0, 0.0000144337d0,
     &                 0.0000104958d0, 0.0000100705d0, 0.0000074034d0,
     &                 0.0000054909d0, 0.0000041189d0, 0.0000040598d0,
     &                 0.0000030289d0, 0.0000030085d0, 0.0000029895d0,
     &                 0.0000023332d0, 0.0000018003d0, 0.0000013931d0,
     &                 0.0000018829d0, 0.0000014630d0, 0.0000006967d0,
     &                 0.0000009205d0, 0.0000012416d0, 0.0000007812d0,
     &                 0.0000010583d0, 0.0000006536d0, 0.0000009038d0,
     &                 0.0000004477d0, 0.0000010522d0, 0.0000006597d0,
     &                 0.0000005567d0, 0.0000004616d0, 0.0000006575d0,
     &                 0.0000003344d0, 0.0000002766d0, 0.0000003963d0,
     &                 0.0000005688d0, 0.0000001793d0, 0.0000004153d0,
     &                 0.0000004620d0, 0.0000003992d0, 0.0000004404d0,
     &                 0.0000002976d0, 0.0000004364d0, 0.0000003740d0 /
      data aij(35,:) / 0.0000430595d0, 0.0000397799d0, 0.0000271616d0,
     &                 0.0000143472d0, 0.0000176142d0, 0.0000168235d0,
     &                 0.0000093500d0, 0.0000067329d0, 0.0000066446d0,
     &                 0.0000082213d0, 0.0000061390d0, 0.0000046372d0,
     &                 0.0000034811d0, 0.0000034735d0, 0.0000026730d0,
     &                 0.0000026634d0, 0.0000015581d0, 0.0000021099d0,
     &                 0.0000012381d0, 0.0000016679d0, 0.0000010186d0,
     &                 0.0000010483d0, 0.0000008241d0, 0.0000011290d0,
     &                 0.0000005475d0, 0.0000005724d0, 0.0000004661d0,
     &                 0.0000006409d0, 0.0000004075d0, 0.0000005660d0,
     &                 0.0000004676d0, 0.0000006678d0, 0.0000005536d0,
     &                 0.0000004588d0, 0.0000003086d0, 0.0000002600d0,
     &                 0.0000002221d0, 0.0000004089d0, 0.0000005910d0,
     &                 0.0000002319d0, 0.0000003286d0, 0.0000002849d0,
     &                 0.0000003188d0, 0.0000001718d0, 0.0000003117d0 /
      data aij(36,:) / 0.0000503956d0, 0.0000262755d0, 0.0000184690d0,
     &                 0.0000169051d0, 0.0000157560d0, 0.0000113206d0,
     &                 0.0000108187d0, 0.0000079141d0, 0.0000075536d0,
     &                 0.0000055140d0, 0.0000054540d0, 0.0000040614d0,
     &                 0.0000039965d0, 0.0000023509d0, 0.0000030468d0,
     &                 0.0000018116d0, 0.0000023451d0, 0.0000013890d0,
     &                 0.0000014094d0, 0.0000011387d0, 0.0000011426d0,
     &                 0.0000007091d0, 0.0000009704d0, 0.0000007647d0,
     &                 0.0000008029d0, 0.0000011008d0, 0.0000003155d0,
     &                 0.0000004312d0, 0.0000004596d0, 0.0000004817d0,
     &                 0.0000005269d0, 0.0000005676d0, 0.0000003706d0,
     &                 0.0000004006d0, 0.0000004315d0, 0.0000004775d0,
     &                 0.0000003098d0, 0.0000003407d0, 0.0000004886d0,
     &                 0.0000004152d0, 0.0000001682d0, 0.0000003093d0,
     &                 0.0000002068d0, 0.0000002292d0, 0.0000002008d0 /
      data aij(37,:) / 0.0000334173d0, 0.0000309726d0, 0.0000218877d0,
     &                 0.0000196442d0, 0.0000138546d0, 0.0000130794d0,
     &                 0.0000094927d0, 0.0000068714d0, 0.0000086744d0,
     &                 0.0000063898d0, 0.0000037817d0, 0.0000048166d0,
     &                 0.0000046345d0, 0.0000026610d0, 0.0000015855d0,
     &                 0.0000020759d0, 0.0000027141d0, 0.0000020849d0,
     &                 0.0000016433d0, 0.0000012979d0, 0.0000007813d0,
     &                 0.0000010596d0, 0.0000004958d0, 0.0000008546d0,
     &                 0.0000008928d0, 0.0000005593d0, 0.0000009962d0,
     &                 0.0000006205d0, 0.0000006694d0, 0.0000004270d0,
     &                 0.0000004437d0, 0.0000003712d0, 0.0000005299d0,
     &                 0.0000003383d0, 0.0000004823d0, 0.0000002425d0,
     &                 0.0000004443d0, 0.0000004843d0, 0.0000004110d0,
     &                 0.0000001667d0, 0.0000001831d0, 0.0000001992d0,
     &                 0.0000002251d0, 0.0000003165d0, 0.0000001686d0 /
      data aij(38,:) / 0.0000389225d0, 0.0000356628d0, 0.0000246098d0,
     &                 0.0000234810d0, 0.0000158826d0, 0.0000154517d0,
     &                 0.0000109594d0, 0.0000061715d0, 0.0000077203d0,
     &                 0.0000057494d0, 0.0000072006d0, 0.0000031528d0,
     &                 0.0000032123d0, 0.0000024479d0, 0.0000023896d0,
     &                 0.0000023499d0, 0.0000018589d0, 0.0000010985d0,
     &                 0.0000014288d0, 0.0000011269d0, 0.0000015138d0,
     &                 0.0000011720d0, 0.0000009382d0, 0.0000009879d0,
     &                 0.0000006065d0, 0.0000008389d0, 0.0000005190d0,
     &                 0.0000007012d0, 0.0000005724d0, 0.0000004774d0,
     &                 0.0000003925d0, 0.0000005365d0, 0.0000002703d0,
     &                 0.0000004855d0, 0.0000004093d0, 0.0000003496d0,
     &                 0.0000003709d0, 0.0000002481d0, 0.0000002089d0,
     &                 0.0000001810d0, 0.0000001978d0, 0.0000001719d0,
     &                 0.0000003092d0, 0.0000001622d0, 0.0000002294d0 /
      data aij(39,:) / 0.0000337803d0, 0.0000404714d0, 0.0000284070d0,
     &                 0.0000198125d0, 0.0000183754d0, 0.0000174965d0,
     &                 0.0000126377d0, 0.0000121499d0, 0.0000086656d0,
     &                 0.0000051637d0, 0.0000063642d0, 0.0000036800d0,
     &                 0.0000027442d0, 0.0000046261d0, 0.0000027739d0,
     &                 0.0000020991d0, 0.0000027411d0, 0.0000015953d0,
     &                 0.0000016454d0, 0.0000010057d0, 0.0000013415d0,
     &                 0.0000010441d0, 0.0000008211d0, 0.0000011139d0,
     &                 0.0000004134d0, 0.0000004295d0, 0.0000005796d0,
     &                 0.0000003604d0, 0.0000004939d0, 0.0000003153d0,
     &                 0.0000004360d0, 0.0000004611d0, 0.0000003900d0,
     &                 0.0000005448d0, 0.0000003467d0, 0.0000004915d0,
     &                 0.0000002450d0, 0.0000002082d0, 0.0000002350d0,
     &                 0.0000002486d0, 0.0000002168d0, 0.0000002377d0,
     &                 0.0000001597d0, 0.0000002236d0, 0.0000001948d0 /
      data aij(40,:) / 0.0000513067d0, 0.0000471119d0, 0.0000328458d0,
     &                 0.0000132586d0, 0.0000211621d0, 0.0000154592d0,
     &                 0.0000085830d0, 0.0000106377d0, 0.0000060140d0,
     &                 0.0000058815d0, 0.0000073820d0, 0.0000042029d0,
     &                 0.0000054567d0, 0.0000041212d0, 0.0000018646d0,
     &                 0.0000023994d0, 0.0000014275d0, 0.0000018493d0,
     &                 0.0000011151d0, 0.0000011543d0, 0.0000009245d0,
     &                 0.0000012066d0, 0.0000007246d0, 0.0000007519d0,
     &                 0.0000004803d0, 0.0000008191d0, 0.0000005011d0,
     &                 0.0000004138d0, 0.0000004368d0, 0.0000003569d0,
     &                 0.0000006387d0, 0.0000005261d0, 0.0000004315d0,
     &                 0.0000002781d0, 0.0000003871d0, 0.0000001946d0,
     &                 0.0000003503d0, 0.0000002314d0, 0.0000002482d0,
     &                 0.0000002172d0, 0.0000003055d0, 0.0000001223d0,
     &                 0.0000002858d0, 0.0000001905d0, 0.0000002117d0 /
      data aij(41,:) / 0.0000450816d0, 0.0000408154d0, 0.0000282845d0,
     &                 0.0000155178d0, 0.0000138761d0, 0.0000175937d0,
     &                 0.0000098262d0, 0.0000121258d0, 0.0000069414d0,
     &                 0.0000066229d0, 0.0000029233d0, 0.0000048194d0,
     &                 0.0000037116d0, 0.0000036148d0, 0.0000027563d0,
     &                 0.0000016149d0, 0.0000009741d0, 0.0000012561d0,
     &                 0.0000012827d0, 0.0000016831d0, 0.0000010254d0,
     &                 0.0000010378d0, 0.0000014009d0, 0.0000008403d0,
     &                 0.0000008877d0, 0.0000009243d0, 0.0000009675d0,
     &                 0.0000003626d0, 0.0000004818d0, 0.0000003920d0,
     &                 0.0000003302d0, 0.0000004548d0, 0.0000002250d0,
     &                 0.0000003083d0, 0.0000003321d0, 0.0000003601d0,
     &                 0.0000001838d0, 0.0000002508d0, 0.0000001319d0,
     &                 0.0000003035d0, 0.0000000966d0, 0.0000001703d0,
     &                 0.0000002389d0, 0.0000002086d0, 0.0000000873d0 /
      data aij(42,:) / 0.0000516615d0, 0.0000359046d0, 0.0000250255d0,
     &                 0.0000176967d0, 0.0000162043d0, 0.0000117023d0,
     &                 0.0000111500d0, 0.0000083372d0, 0.0000080845d0,
     &                 0.0000057760d0, 0.0000057683d0, 0.0000042198d0,
     &                 0.0000042307d0, 0.0000031731d0, 0.0000018613d0,
     &                 0.0000018786d0, 0.0000018362d0, 0.0000018880d0,
     &                 0.0000011419d0, 0.0000019392d0, 0.0000015208d0,
     &                 0.0000005532d0, 0.0000005717d0, 0.0000005781d0,
     &                 0.0000007821d0, 0.0000004847d0, 0.0000005016d0,
     &                 0.0000004026d0, 0.0000005570d0, 0.0000007553d0,
     &                 0.0000003651d0, 0.0000002375d0, 0.0000004137d0,
     &                 0.0000004454d0, 0.0000003721d0, 0.0000003077d0,
     &                 0.0000002562d0, 0.0000001692d0, 0.0000001437d0,
     &                 0.0000003380d0, 0.0000001705d0, 0.0000002396d0,
     &                 0.0000001596d0, 0.0000002284d0, 0.0000001911d0 /
      data aij(43,:) / 0.0000345441d0, 0.0000316241d0, 0.0000286584d0,
     &                 0.0000206648d0, 0.0000186795d0, 0.0000136872d0,
     &                 0.0000097047d0, 0.0000096665d0, 0.0000091419d0,
     &                 0.0000067531d0, 0.0000038985d0, 0.0000048445d0,
     &                 0.0000028978d0, 0.0000037146d0, 0.0000021550d0,
     &                 0.0000021547d0, 0.0000016273d0, 0.0000009960d0,
     &                 0.0000012967d0, 0.0000013308d0, 0.0000008052d0,
     &                 0.0000010640d0, 0.0000008348d0, 0.0000008528d0,
     &                 0.0000008847d0, 0.0000003357d0, 0.0000004374d0,
     &                 0.0000005905d0, 0.0000002889d0, 0.0000005062d0,
     &                 0.0000005427d0, 0.0000003357d0, 0.0000003624d0,
     &                 0.0000002976d0, 0.0000004179d0, 0.0000002635d0,
     &                 0.0000003711d0, 0.0000001854d0, 0.0000003373d0,
     &                 0.0000001698d0, 0.0000002400d0, 0.0000001599d0,
     &                 0.0000001739d0, 0.0000001177d0, 0.0000002124d0 /
      data aij(44,:) / 0.0000408929d0, 0.0000365974d0, 0.0000338537d0,
     &                 0.0000239837d0, 0.0000219789d0, 0.0000159079d0,
     &                 0.0000148083d0, 0.0000064093d0, 0.0000047381d0,
     &                 0.0000060372d0, 0.0000058068d0, 0.0000056862d0,
     &                 0.0000032968d0, 0.0000024911d0, 0.0000024593d0,
     &                 0.0000019073d0, 0.0000019128d0, 0.0000019005d0,
     &                 0.0000014645d0, 0.0000009027d0, 0.0000008974d0,
     &                 0.0000012109d0, 0.0000007360d0, 0.0000005796d0,
     &                 0.0000007748d0, 0.0000003802d0, 0.0000005085d0,
     &                 0.0000004071d0, 0.0000004279d0, 0.0000004439d0,
     &                 0.0000003642d0, 0.0000002963d0, 0.0000003129d0,
     &                 0.0000003366d0, 0.0000003574d0, 0.0000001793d0,
     &                 0.0000001519d0, 0.0000002667d0, 0.0000001760d0,
     &                 0.0000002482d0, 0.0000001261d0, 0.0000002263d0,
     &                 0.0000001493d0, 0.0000002143d0, 0.0000000869d0 /
      data aij(45,:) / 0.0000460894d0, 0.0000423271d0, 0.0000297879d0,
     &                 0.0000205531d0, 0.0000193030d0, 0.0000178901d0,
     &                 0.0000130914d0, 0.0000096919d0, 0.0000091541d0,
     &                 0.0000052562d0, 0.0000039234d0, 0.0000038485d0,
     &                 0.0000029218d0, 0.0000028777d0, 0.0000021609d0,
     &                 0.0000021577d0, 0.0000022213d0, 0.0000012807d0,
     &                 0.0000010154d0, 0.0000007967d0, 0.0000006264d0,
     &                 0.0000008077d0, 0.0000006542d0, 0.0000006707d0,
     &                 0.0000006786d0, 0.0000005617d0, 0.0000004357d0,
     &                 0.0000005927d0, 0.0000004775d0, 0.0000003001d0,
     &                 0.0000003137d0, 0.0000002632d0, 0.0000002739d0,
     &                 0.0000002881d0, 0.0000003141d0, 0.0000002009d0,
     &                 0.0000002152d0, 0.0000003873d0, 0.0000001959d0,
     &                 0.0000001287d0, 0.0000002305d0, 0.0000001938d0,
     &                 0.0000002115d0, 0.0000003004d0, 0.0000001533d0 /
c
c     Bij neck correction constants
c
      data bij(1,:)  / -1.20d0, -1.05d0, -1.30d0, -1.15d0, -1.00d0,
     &                 -0.85d0, -0.90d0, -0.75d0, -1.20d0, -1.05d0,
     &                 -0.90d0, -0.95d0, -0.80d0, -0.65d0, -0.70d0,
     &                 -0.75d0, -0.60d0, -0.45d0, -0.50d0, -0.75d0,
     &                 -0.40d0, -0.25d0, -0.30d0, -0.15d0, -0.20d0,
     &                 -0.25d0,  0.10d0,  0.05d0,  0.20d0, -0.05d0,
     &                  0.30d0,  0.25d0,  0.40d0,  0.15d0,  0.10d0,
     &                  0.25d0,  0.60d0,  0.55d0,  0.70d0,  0.45d0,
     &                  0.80d0,  0.55d0,  0.90d0,  0.65d0,  1.00d0 /
      data bij(2,:)  / -1.25d0, -1.10d0, -0.95d0, -1.40d0, -1.05d0,
     &                 -1.10d0, -0.95d0, -0.80d0, -0.85d0, -0.70d0,
     &                 -0.55d0, -1.00d0, -0.65d0, -0.70d0, -0.95d0,
     &                 -0.60d0, -0.65d0, -0.70d0, -0.55d0, -0.60d0,
     &                 -0.45d0, -0.10d0, -0.15d0,  0.00d0, -0.05d0,
     &                  0.10d0,  0.05d0,  0.20d0,  0.15d0,  0.10d0,
     &                  0.25d0,  0.00d0,  0.15d0, -0.10d0,  0.25d0,
     &                  0.20d0,  0.35d0,  0.50d0,  0.65d0,  0.40d0,
     &                  0.75d0,  0.70d0,  0.85d0,  0.60d0,  0.95d0 /
      data bij(3,:)  / -1.30d0, -1.15d0, -1.00d0, -1.05d0, -0.90d0,
     &                 -0.95d0, -1.20d0, -0.85d0, -0.90d0, -0.95d0,
     &                 -0.80d0, -0.65d0, -0.70d0, -0.75d0, -0.40d0,
     &                 -0.65d0, -0.30d0, -0.35d0, -0.60d0, -0.45d0,
     &                 -0.50d0, -0.35d0, -0.20d0, -0.25d0, -0.10d0,
     &                  0.05d0,  0.00d0,  0.15d0, -0.10d0,  0.25d0,
     &                  0.00d0,  0.35d0,  0.10d0,  0.45d0,  0.40d0,
     &                  0.55d0,  0.30d0,  0.65d0,  0.60d0,  0.75d0,
     &                  0.50d0,  0.85d0,  0.80d0,  0.95d0,  0.70d0 /
      data bij(4,:)  / -1.15d0, -1.40d0, -1.05d0, -0.90d0, -1.15d0,
     &                 -0.80d0, -1.05d0, -0.90d0, -0.95d0, -0.60d0,
     &                 -0.85d0, -0.70d0, -0.75d0, -0.40d0, -0.45d0,
     &                 -0.30d0, -0.35d0, -0.40d0, -0.25d0, -0.30d0,
     &                 -0.15d0, -0.40d0, -0.05d0, -0.30d0, -0.15d0,
     &                 -0.20d0, -0.05d0,  0.10d0,  0.25d0,  0.00d0,
     &                  0.35d0,  0.10d0,  0.25d0,  0.20d0,  0.35d0,
     &                  0.30d0,  0.45d0,  0.40d0,  0.75d0,  0.50d0,
     &                  0.65d0,  0.80d0,  0.75d0,  0.90d0,  0.65d0 /
      data bij(5,:)  / -1.00d0, -1.25d0, -0.90d0, -1.15d0, -1.00d0,
     &                 -0.85d0, -0.90d0, -0.75d0, -0.60d0, -0.85d0,
     &                 -0.50d0, -0.95d0, -0.80d0, -0.65d0, -0.70d0,
     &                 -0.55d0, -0.60d0, -0.25d0, -0.50d0, -0.35d0,
     &                 -0.40d0, -0.05d0, -0.10d0,  0.05d0, -0.20d0,
     &                  0.15d0, -0.10d0,  0.05d0,  0.00d0,  0.15d0,
     &                  0.10d0,  0.25d0,  0.40d0,  0.35d0,  0.50d0,
     &                  0.45d0,  0.60d0,  0.55d0,  0.70d0,  0.45d0,
     &                  0.80d0,  0.75d0,  0.90d0,  0.65d0,  1.00d0 /
      data bij(6,:)  / -0.85d0, -1.10d0, -1.15d0, -1.00d0, -0.85d0,
     &                 -0.90d0, -0.95d0, -0.80d0, -0.65d0, -0.50d0,
     &                 -0.55d0, -0.60d0, -0.65d0, -0.50d0, -0.55d0,
     &                 -0.40d0, -0.25d0, -0.50d0, -0.15d0, -0.60d0,
     &                 -0.05d0, -0.30d0, -0.15d0,  0.00d0, -0.05d0,
     &                  0.10d0,  0.05d0,  0.20d0,  0.15d0,  0.30d0,
     &                  0.05d0,  0.40d0,  0.35d0,  0.50d0,  0.45d0,
     &                  0.60d0,  0.55d0,  0.70d0,  0.45d0,  0.80d0,
     &                  0.75d0,  0.90d0,  0.85d0,  1.00d0,  0.75d0 /
      data bij(7,:)  / -0.90d0, -1.15d0, -1.00d0, -1.05d0, -0.90d0,
     &                 -0.55d0, -0.60d0, -0.45d0, -0.70d0, -0.55d0,
     &                 -0.60d0, -0.65d0, -0.50d0, -0.35d0, -0.40d0,
     &                 -0.25d0, -0.50d0, -0.35d0, -0.20d0, -0.05d0,
     &                 -0.10d0,  0.05d0,  0.00d0,  0.15d0,  0.30d0,
     &                  0.25d0,  0.40d0,  0.15d0,  0.30d0, -0.15d0,
     &                  0.40d0,  0.55d0,  0.50d0,  0.25d0,  0.60d0,
     &                  0.55d0,  0.90d0,  0.45d0,  0.60d0,  0.75d0,
     &                  0.70d0,  0.85d0,  1.00d0,  0.95d0,  0.90d0 /
      data bij(8,:)  / -1.15d0, -1.00d0, -1.25d0, -0.90d0, -0.75d0,
     &                 -0.80d0, -0.45d0, -0.50d0, -0.95d0, -0.40d0,
     &                 -0.45d0, -0.70d0, -0.55d0, -0.40d0, -0.45d0,
     &                 -0.30d0, -0.15d0, -0.20d0, -0.45d0, -0.10d0,
     &                 -0.15d0, -0.20d0, -0.05d0,  0.10d0,  0.05d0,
     &                  0.20d0,  0.15d0,  0.10d0,  0.05d0,  0.40d0,
     &                  0.35d0,  0.50d0,  0.45d0,  0.20d0,  0.55d0,
     &                  0.50d0,  0.45d0,  0.20d0,  0.75d0,  0.70d0,
     &                  0.85d0,  0.80d0,  0.95d0,  1.10d0,  1.05d0 /
      data bij(9,:)  / -1.00d0, -0.85d0, -0.90d0, -0.75d0, -0.80d0,
     &                 -0.85d0, -0.90d0, -0.55d0, -0.60d0, -0.45d0,
     &                 -0.50d0, -0.35d0, -0.40d0, -0.65d0, -0.30d0,
     &                 -0.35d0,  0.00d0, -0.25d0,  0.10d0, -0.35d0,
     &                  0.00d0, -0.05d0, -0.10d0,  0.05d0,  0.00d0,
     &                  0.15d0,  0.10d0,  0.25d0,  0.20d0,  0.15d0,
     &                  0.30d0,  0.45d0,  0.00d0,  0.55d0,  0.70d0,
     &                  0.65d0,  0.80d0,  0.75d0,  0.90d0,  1.05d0,
     &                  1.00d0,  1.15d0,  1.10d0,  1.25d0,  1.00d0 /
      data bij(10,:) / -1.05d0, -1.10d0, -0.75d0, -0.60d0, -0.65d0,
     &                 -0.70d0, -0.95d0, -0.60d0, -0.45d0, -0.70d0,
     &                 -0.55d0, -0.20d0, -0.45d0, -0.30d0, -0.35d0,
     &                 -0.20d0, -0.25d0, -0.10d0,  0.05d0,  0.00d0,
     &                 -0.05d0,  0.30d0,  0.25d0,  0.00d0,  0.15d0,
     &                  0.10d0,  0.25d0,  0.20d0,  0.55d0,  0.50d0,
     &                  0.45d0,  0.20d0,  0.75d0,  0.30d0,  0.85d0,
     &                  0.60d0,  0.55d0,  0.70d0,  0.85d0,  0.80d0,
     &                  0.75d0,  0.90d0,  1.05d0,  1.00d0,  1.35d0 /
      data bij(11,:) / -1.10d0, -0.75d0, -0.80d0, -0.65d0, -0.50d0,
     &                 -0.75d0, -0.60d0, -0.45d0, -0.70d0, -0.55d0,
     &                 -0.40d0, -0.65d0, -0.30d0, -0.15d0, -0.20d0,
     &                 -0.25d0, -0.30d0, -0.15d0, -0.20d0, -0.05d0,
     &                 -0.10d0,  0.05d0,  0.20d0,  0.15d0,  0.30d0,
     &                  0.05d0,  0.20d0,  0.55d0,  0.30d0,  0.45d0,
     &                  0.40d0,  0.55d0,  0.50d0,  0.65d0,  0.60d0,
     &                  0.55d0,  0.90d0,  0.65d0,  1.00d0,  0.95d0,
     &                  0.90d0,  0.65d0,  1.00d0,  0.75d0,  1.10d0 /
      data bij(12,:) / -0.75d0, -1.00d0, -0.65d0, -0.50d0, -0.75d0,
     &                 -0.60d0, -0.65d0, -0.30d0, -0.35d0, -0.20d0,
     &                 -0.65d0, -0.50d0, -0.55d0, -0.20d0, -0.05d0,
     &                  0.10d0, -0.15d0, -0.20d0, -0.25d0, -0.10d0,
     &                 -0.15d0,  0.00d0,  0.15d0,  0.10d0,  0.25d0,
     &                  0.20d0,  0.35d0,  0.30d0,  0.25d0,  0.40d0,
     &                  0.55d0,  0.70d0,  0.45d0,  0.60d0,  0.55d0,
     &                  0.90d0,  0.85d0,  1.00d0,  0.75d0,  1.10d0,
     &                  1.05d0,  1.20d0,  1.15d0,  1.30d0,  1.05d0 /
      data bij(13,:) / -0.80d0, -0.65d0, -0.70d0, -0.75d0, -0.60d0,
     &                 -0.65d0, -0.50d0, -0.55d0, -0.40d0, -0.65d0,
     &                 -0.30d0, -0.55d0, -0.40d0, -0.45d0, -0.10d0,
     &                 -0.35d0,  0.00d0,  0.15d0,  0.10d0,  0.05d0,
     &                 -0.20d0,  0.35d0, -0.10d0,  0.05d0,  0.20d0,
     &                  0.35d0,  0.50d0,  0.45d0,  0.40d0,  0.75d0,
     &                  0.50d0,  0.85d0,  0.60d0,  0.55d0,  0.70d0,
     &                  0.85d0,  1.00d0,  0.75d0,  1.10d0,  1.05d0,
     &                  1.00d0,  1.15d0,  0.90d0,  1.05d0,  1.00d0 /
      data bij(14,:) / -0.85d0, -0.70d0, -0.95d0, -0.60d0, -0.85d0,
     &                 -0.50d0, -0.75d0, -0.40d0, -0.25d0, -0.30d0,
     &                 -0.15d0, -0.20d0, -0.25d0, -0.10d0,  0.05d0,
     &                 -0.20d0, -0.05d0,  0.10d0,  0.05d0, -0.40d0,
     &                  0.35d0,  0.30d0,  0.45d0,  0.20d0, -0.05d0,
     &                  0.10d0,  0.25d0,  0.40d0,  0.35d0,  0.50d0,
     &                  0.65d0,  1.00d0,  0.75d0,  0.90d0,  0.65d0,
     &                  0.20d0,  0.75d0,  0.70d0,  0.85d0,  1.00d0,
     &                  0.95d0,  1.10d0,  1.25d0,  1.20d0,  1.35d0 /
      data bij(15,:) / -0.70d0, -0.95d0, -0.60d0, -0.65d0, -0.50d0,
     &                 -0.55d0, -0.40d0, -0.25d0, -0.30d0, -0.15d0,
     &                 -0.20d0, -0.05d0, -0.30d0,  0.05d0,  0.20d0,
     &                 -0.05d0, -0.30d0,  0.05d0,  0.00d0,  0.15d0,
     &                  0.10d0,  0.25d0,  0.20d0,  0.35d0,  0.10d0,
     &                  0.45d0,  0.60d0,  0.55d0,  0.50d0,  0.25d0,
     &                  0.20d0,  0.55d0,  0.70d0,  0.85d0,  0.80d0,
     &                  1.15d0,  0.90d0,  1.25d0,  0.80d0,  0.95d0,
     &                  0.90d0,  1.05d0,  1.00d0,  1.55d0,  0.90d0 /
      data bij(16,:) / -0.35d0, -0.80d0, -0.45d0, -0.50d0, -0.35d0,
     &                 -0.40d0, -0.25d0, -0.50d0, -0.15d0, -0.20d0,
     &                 -0.05d0, -0.50d0, -0.15d0, -0.40d0, -0.05d0,
     &                 -0.10d0,  0.25d0,  0.20d0,  0.15d0,  0.10d0,
     &                  0.25d0,  0.20d0,  0.35d0,  0.30d0,  0.45d0,
     &                  0.40d0,  0.35d0,  0.70d0,  0.65d0,  0.40d0,
     &                  0.75d0,  0.50d0,  0.65d0,  0.60d0,  0.95d0,
     &                  0.90d0,  1.05d0,  1.00d0,  1.15d0,  0.70d0,
     &                  1.25d0,  1.00d0,  0.95d0,  1.10d0,  1.05d0 /
      data bij(17,:) / -0.40d0, -0.45d0, -0.50d0, -0.35d0, -0.60d0,
     &                 -0.45d0, -0.50d0, -0.15d0,  0.00d0, -0.25d0,
     &                 -0.30d0, -0.15d0, -0.40d0, -0.05d0,  0.10d0,
     &                  0.25d0,  0.40d0,  0.15d0,  0.10d0,  0.05d0,
     &                  0.40d0, -0.05d0,  0.10d0,  0.25d0,  0.40d0,
     &                  0.35d0,  0.50d0,  0.45d0,  0.60d0,  0.75d0,
     &                  0.90d0,  0.85d0,  0.60d0,  0.95d0,  0.70d0,
     &                  1.05d0,  0.80d0,  0.95d0,  0.90d0,  1.05d0,
     &                  1.20d0,  1.15d0,  1.30d0,  1.25d0,  1.40d0 /
      data bij(18,:) / -0.45d0, -0.30d0, -0.35d0, -0.40d0, -0.25d0,
     &                 -0.10d0, -0.35d0, -0.20d0, -0.25d0, -0.30d0,
     &                 -0.15d0,  0.00d0, -0.05d0,  0.10d0,  0.25d0,
     &                  0.20d0,  0.15d0,  0.30d0,  0.05d0,  0.20d0,
     &                  0.35d0,  0.50d0,  0.65d0,  0.20d0,  0.55d0,
     &                  0.50d0,  0.85d0,  0.40d0,  0.75d0,  0.50d0,
     &                  0.65d0,  0.60d0,  0.75d0,  0.90d0,  1.05d0,
     &                  1.20d0,  0.95d0,  1.30d0,  1.05d0,  1.00d0,
     &                  0.95d0,  1.30d0,  1.25d0,  1.40d0,  0.95d0 /
      data bij(19,:) / -0.30d0, -0.35d0, -0.40d0, -0.45d0, -0.30d0,
     &                 -0.35d0, -0.20d0, -0.45d0, -0.10d0,  0.05d0,
     &                  0.20d0, -0.05d0, -0.10d0, -0.15d0,  0.20d0,
     &                  0.15d0,  0.10d0,  0.05d0,  0.60d0,  0.35d0,
     &                  0.50d0,  0.25d0,  0.40d0,  0.55d0,  0.70d0,
     &                  0.65d0,  0.80d0,  0.55d0,  0.70d0,  1.05d0,
     &                  1.00d0,  0.55d0,  0.90d0,  1.05d0,  1.20d0,
     &                  0.75d0,  1.10d0,  0.85d0,  1.40d0,  1.35d0,
     &                  1.50d0,  1.05d0,  1.20d0,  1.15d0,  1.30d0 /
      data bij(20,:) / -0.55d0, -0.40d0, -0.65d0, -0.30d0, -0.35d0,
     &                 -0.20d0, -0.25d0, -0.10d0, -0.35d0,  0.00d0,
     &                 -0.05d0,  0.10d0,  0.05d0,  0.00d0, -0.25d0,
     &                  0.30d0,  0.25d0,  0.20d0,  0.35d0, -0.10d0,
     &                  0.25d0,  0.40d0,  0.55d0,  0.70d0,  0.65d0,
     &                  0.60d0,  0.75d0,  0.90d0,  0.65d0,  0.40d0,
     &                  0.95d0,  0.70d0,  1.05d0,  1.00d0,  1.15d0,
     &                  0.90d0,  1.05d0,  1.00d0,  0.75d0,  1.30d0,
     &                  1.25d0,  1.60d0,  1.35d0,  1.10d0,  1.45d0 /
      data bij(21,:) / -0.40d0, -0.45d0, -0.70d0, -0.15d0, -0.20d0,
     &                 -0.05d0,  0.10d0, -0.35d0,  0.00d0,  0.15d0,
     &                 -0.10d0,  0.25d0,  0.00d0,  0.35d0,  0.30d0,
     &                  0.25d0,  0.20d0,  0.55d0,  0.50d0,  0.45d0,
     &                  0.20d0,  0.75d0,  0.50d0,  0.65d0,  0.60d0,
     &                  0.75d0,  0.50d0,  0.85d0,  0.80d0,  0.95d0,
     &                  1.10d0,  1.05d0,  1.00d0,  1.15d0,  1.10d0,
     &                  1.05d0,  1.20d0,  0.95d0,  1.10d0,  1.25d0,
     &                  1.20d0,  1.35d0,  1.30d0,  1.45d0,  1.40d0 /
      data bij(22,:) / -0.25d0, -0.10d0, -0.15d0, -0.40d0, -0.45d0,
     &                 -0.30d0,  0.05d0, -0.20d0,  0.15d0,  0.10d0,
     &                  0.05d0,  0.00d0,  0.35d0,  0.10d0,  0.45d0,
     &                  0.20d0,  0.35d0,  0.10d0,  0.25d0,  0.40d0,
     &                  0.55d0,  0.30d0,  0.65d0,  0.20d0,  0.35d0,
     &                  0.30d0,  0.65d0,  0.60d0,  0.75d0,  0.90d0,
     &                  0.85d0,  1.00d0,  0.95d0,  1.30d0,  1.05d0,
     &                  1.20d0,  1.15d0,  1.10d0,  1.05d0,  1.20d0,
     &                  1.35d0,  1.30d0,  1.45d0,  1.40d0,  1.75d0 /
      data bij(23,:) / -0.30d0, -0.15d0, -0.20d0, -0.25d0,  0.10d0,
     &                 -0.35d0, -0.20d0,  0.15d0, -0.10d0,  0.05d0,
     &                  0.40d0,  0.15d0,  0.10d0,  0.45d0,  0.00d0,
     &                  0.35d0,  0.10d0,  0.45d0,  0.60d0,  0.55d0,
     &                  0.50d0,  0.65d0,  0.20d0,  0.35d0,  0.90d0,
     &                  0.45d0,  0.60d0,  0.35d0,  0.90d0,  0.65d0,
     &                  0.60d0,  0.95d0,  1.10d0,  1.25d0,  1.00d0,
     &                  1.15d0,  1.30d0,  1.45d0,  1.00d0,  1.15d0,
     &                  1.50d0,  1.65d0,  1.20d0,  1.35d0,  1.30d0 /
      data bij(24,:) / -0.15d0, -0.20d0, -0.05d0, -0.10d0, -0.15d0,
     &                 -0.40d0, -0.05d0,  0.10d0,  0.05d0,  0.00d0,
     &                  0.15d0,  0.30d0,  0.25d0,  0.20d0,  0.55d0,
     &                  0.50d0,  0.45d0,  0.20d0,  0.15d0,  0.70d0,
     &                  0.45d0,  0.60d0,  0.35d0,  0.90d0,  0.65d0,
     &                  0.80d0,  0.55d0,  0.50d0,  0.45d0,  0.80d0,
     &                  0.95d0,  0.90d0,  1.05d0,  1.00d0,  0.95d0,
     &                  1.10d0,  1.25d0,  1.40d0,  1.35d0,  1.30d0,
     &                  1.45d0,  1.20d0,  1.55d0,  1.70d0,  1.65d0 /
      data bij(25,:) /  0.00d0, -0.25d0, -0.10d0, -0.15d0,  0.00d0,
     &                 -0.05d0, -0.10d0,  0.05d0,  0.00d0,  0.15d0,
     &                  0.30d0,  0.25d0,  0.20d0,  0.15d0,  0.50d0,
     &                  0.45d0,  0.20d0,  0.35d0,  0.70d0,  0.45d0,
     &                  0.40d0,  0.35d0,  0.70d0,  0.65d0,  0.80d0,
     &                  0.75d0,  0.90d0,  0.65d0,  1.00d0,  0.95d0,
     &                  1.30d0,  1.25d0,  1.00d0,  1.35d0,  0.90d0,
     &                  1.25d0,  1.00d0,  1.35d0,  1.30d0,  1.45d0,
     &                  1.40d0,  1.55d0,  1.70d0,  1.25d0,  1.60d0 /
      data bij(26,:) / -0.25d0, -0.10d0, -0.15d0,  0.00d0,  0.15d0,
     &                  0.10d0, -0.15d0,  0.20d0,  0.15d0,  0.10d0,
     &                  0.05d0,  0.40d0,  0.35d0,  0.50d0,  0.05d0,
     &                  0.00d0,  0.35d0,  0.50d0,  0.25d0,  0.60d0,
     &                  0.35d0,  0.50d0,  0.65d0,  1.00d0,  0.95d0,
     &                  0.90d0,  0.85d0,  0.80d0,  1.15d0,  1.10d0,
     &                  1.25d0,  1.40d0,  1.15d0,  0.90d0,  1.05d0,
     &                  1.60d0,  1.55d0,  1.10d0,  1.25d0,  1.60d0,
     &                  1.35d0,  1.30d0,  1.45d0,  1.60d0,  1.55d0 /
      data bij(27,:) / -0.10d0, -0.15d0,  0.00d0, -0.05d0,  0.10d0,
     &                 -0.15d0,  0.00d0,  0.35d0,  0.30d0,  0.05d0,
     &                  0.20d0,  0.55d0,  0.30d0,  0.65d0,  0.60d0,
     &                  0.15d0,  0.50d0,  0.25d0,  0.60d0,  0.55d0,
     &                  0.90d0,  0.85d0,  0.60d0,  0.55d0,  0.70d0,
     &                  1.05d0,  0.80d0,  0.95d0,  0.90d0,  1.05d0,
     &                  1.00d0,  1.15d0,  1.30d0,  1.25d0,  1.20d0,
     &                  1.55d0,  0.90d0,  1.45d0,  1.40d0,  1.35d0,
     &                  1.90d0,  1.45d0,  1.60d0,  1.55d0,  1.50d0 /
      data bij(28,:) /  0.05d0,  0.00d0,  0.15d0,  0.10d0, -0.15d0,
     &                  0.20d0,  0.35d0,  0.10d0,  0.45d0,  0.20d0,
     &                  0.15d0,  0.30d0,  0.05d0,  0.40d0,  0.15d0,
     &                  0.70d0,  0.45d0,  0.40d0,  0.75d0,  0.50d0,
     &                  0.85d0,  0.60d0,  0.75d0,  0.90d0,  1.05d0,
     &                  1.20d0,  1.15d0,  0.90d0,  1.25d0,  1.00d0,
     &                  0.95d0,  1.10d0,  1.25d0,  1.00d0,  1.15d0,
     &                  1.10d0,  1.25d0,  1.60d0,  1.15d0,  1.30d0,
     &                  1.25d0,  1.20d0,  1.75d0,  1.50d0,  1.65d0 /
      data bij(29,:) /  0.20d0,  0.15d0, -0.10d0, -0.15d0,  0.00d0,
     &                  0.15d0,  0.10d0,  0.45d0,  0.20d0,  0.35d0,
     &                  0.10d0,  0.45d0,  0.40d0,  0.35d0,  0.30d0,
     &                  0.65d0,  0.40d0,  0.35d0,  0.70d0,  0.85d0,
     &                  0.60d0,  0.55d0,  0.50d0,  0.85d0,  1.20d0,
     &                  0.95d0,  0.50d0,  0.85d0,  1.20d0,  1.35d0,
     &                  0.90d0,  1.05d0,  1.40d0,  1.35d0,  1.50d0,
     &                  1.25d0,  1.40d0,  1.55d0,  1.70d0,  1.05d0,
     &                  1.60d0,  1.75d0,  1.70d0,  1.65d0,  1.80d0 /
      data bij(30,:) /  0.15d0,  0.10d0,  0.05d0,  0.00d0,  0.15d0,
     &                  0.10d0,  0.25d0,  0.60d0,  0.35d0,  0.30d0,
     &                  0.05d0,  0.60d0,  0.55d0,  0.30d0,  0.25d0,
     &                  0.40d0,  0.55d0,  0.50d0,  0.85d0,  0.40d0,
     &                  0.75d0,  1.10d0,  1.05d0,  0.40d0,  0.95d0,
     &                  0.70d0,  0.85d0,  1.00d0,  0.95d0,  1.30d0,
     &                  1.05d0,  1.40d0,  1.35d0,  1.70d0,  1.45d0,
     &                  1.40d0,  1.35d0,  1.50d0,  1.45d0,  1.60d0,
     &                  1.55d0,  1.70d0,  1.85d0,  2.00d0,  1.75d0 /
      data bij(31,:) /  0.10d0,  0.05d0,  0.00d0,  0.15d0,  0.10d0,
     &                  0.25d0,  0.20d0,  0.35d0,  0.10d0,  0.45d0,
     &                  0.60d0,  0.55d0,  0.10d0,  0.65d0,  0.80d0,
     &                  0.75d0,  0.30d0,  0.65d0,  0.40d0,  0.75d0,
     &                  1.10d0,  0.85d0,  0.40d0,  0.75d0,  0.70d0,
     &                  0.85d0,  1.20d0,  0.95d0,  1.30d0,  1.25d0,
     &                  1.40d0,  0.95d0,  1.50d0,  0.85d0,  1.60d0,
     &                  1.75d0,  1.70d0,  0.65d0,  1.60d0,  1.55d0,
     &                  1.50d0,  1.65d0,  1.80d0,  1.75d0,  1.90d0 /
      data bij(32,:) /  0.05d0,  0.20d0,  0.15d0,  0.30d0,  0.25d0,
     &                  0.40d0,  0.35d0,  0.50d0,  0.65d0,  0.20d0,
     &                  0.55d0,  0.30d0,  0.45d0,  0.40d0,  0.55d0,
     &                  0.90d0,  0.85d0,  0.40d0,  0.55d0,  0.90d0,
     &                  0.65d0,  0.80d0,  0.95d0,  1.10d0,  1.45d0,
     &                  1.40d0,  0.95d0,  1.10d0,  1.05d0,  1.00d0,
     &                  1.35d0,  1.30d0,  1.05d0,  1.40d0,  1.75d0,
     &                  1.30d0,  1.65d0,  1.40d0,  1.75d0,  1.50d0,
     &                  2.05d0,  1.60d0,  2.15d0,  1.90d0,  1.65d0 /
      data bij(33,:) /  0.20d0,  0.15d0,  0.30d0,  0.25d0,  0.40d0,
     &                  0.35d0,  0.30d0,  0.25d0,  0.60d0,  0.35d0,
     &                  0.70d0,  0.85d0,  0.20d0,  0.95d0,  0.70d0,
     &                  0.65d0,  0.80d0,  0.55d0,  0.90d0,  0.85d0,
     &                  1.00d0,  0.95d0,  0.90d0,  1.05d0,  1.20d0,
     &                  1.35d0,  1.10d0,  1.25d0,  0.80d0,  1.55d0,
     &                  1.30d0,  1.25d0,  1.20d0,  1.95d0,  1.50d0,
     &                  1.45d0,  1.20d0,  1.95d0,  1.30d0,  1.85d0,
     &                  1.40d0,  1.95d0,  1.50d0,  1.85d0,  1.80d0 /
      data bij(34,:) /  0.35d0,  0.50d0,  0.45d0,  0.40d0,  0.55d0,
     &                  0.50d0,  0.45d0,  0.60d0,  0.55d0,  0.50d0,
     &                  0.45d0,  0.60d0,  0.55d0,  0.70d0,  0.85d0,
     &                  0.80d0,  0.75d0,  0.70d0,  1.05d0,  1.00d0,
     &                  0.55d0,  0.90d0,  1.25d0,  1.00d0,  1.35d0,
     &                  1.10d0,  1.45d0,  1.00d0,  1.75d0,  1.50d0,
     &                  1.45d0,  1.40d0,  1.75d0,  1.30d0,  1.25d0,
     &                  1.60d0,  1.95d0,  1.10d0,  1.85d0,  2.00d0,
     &                  1.95d0,  2.10d0,  1.85d0,  2.20d0,  2.15d0 /
      data bij(35,:) /  0.30d0,  0.45d0,  0.40d0,  0.15d0,  0.50d0,
     &                  0.65d0,  0.40d0,  0.35d0,  0.50d0,  0.85d0,
     &                  0.80d0,  0.75d0,  0.70d0,  0.85d0,  0.80d0,
     &                  0.95d0,  0.70d0,  1.05d0,  0.80d0,  1.15d0,
     &                  0.90d0,  1.05d0,  1.00d0,  1.35d0,  0.90d0,
     &                  1.05d0,  1.00d0,  1.35d0,  1.10d0,  1.45d0,
     &                  1.40d0,  1.75d0,  1.70d0,  1.65d0,  1.40d0,
     &                  1.35d0,  1.30d0,  1.85d0,  2.20d0,  1.55d0,
     &                  1.90d0,  1.85d0,  2.00d0,  1.55d0,  2.10d0 /
      data bij(36,:) /  0.45d0,  0.20d0,  0.15d0,  0.30d0,  0.45d0,
     &                  0.40d0,  0.55d0,  0.50d0,  0.65d0,  0.60d0,
     &                  0.75d0,  0.70d0,  0.85d0,  0.60d0,  0.95d0,
     &                  0.70d0,  1.05d0,  0.80d0,  0.95d0,  0.90d0,
     &                  1.05d0,  0.80d0,  1.15d0,  1.10d0,  1.25d0,
     &                  1.60d0,  0.75d0,  1.10d0,  1.25d0,  1.40d0,
     &                  1.55d0,  1.70d0,  1.45d0,  1.60d0,  1.75d0,
     &                  1.90d0,  1.65d0,  1.80d0,  2.15d0,  2.10d0,
     &                  1.45d0,  2.00d0,  1.75d0,  1.90d0,  1.85d0 /
      data bij(37,:) /  0.20d0,  0.35d0,  0.30d0,  0.45d0,  0.40d0,
     &                  0.55d0,  0.50d0,  0.45d0,  0.80d0,  0.75d0,
     &                  0.50d0,  0.85d0,  1.00d0,  0.75d0,  0.50d0,
     &                  0.85d0,  1.20d0,  1.15d0,  1.10d0,  1.05d0,
     &                  0.80d0,  1.15d0,  0.70d0,  1.25d0,  1.40d0,
     &                  1.15d0,  1.70d0,  1.45d0,  1.60d0,  1.35d0,
     &                  1.50d0,  1.45d0,  1.80d0,  1.55d0,  1.90d0,
     &                  1.45d0,  2.00d0,  2.15d0,  2.10d0,  1.45d0,
     &                  1.60d0,  1.75d0,  1.90d0,  2.25d0,  1.80d0 /
      data bij(38,:) /  0.35d0,  0.50d0,  0.45d0,  0.60d0,  0.55d0,
     &                  0.70d0,  0.65d0,  0.40d0,  0.75d0,  0.70d0,
     &                  1.05d0,  0.60d0,  0.75d0,  0.70d0,  0.85d0,
     &                  1.00d0,  0.95d0,  0.70d0,  1.05d0,  1.00d0,
     &                  1.35d0,  1.30d0,  1.25d0,  1.40d0,  1.15d0,
     &                  1.50d0,  1.25d0,  1.60d0,  1.55d0,  1.50d0,
     &                  1.45d0,  1.80d0,  1.35d0,  1.90d0,  1.85d0,
     &                  1.80d0,  1.95d0,  1.70d0,  1.65d0,  1.60d0,
     &                  1.75d0,  1.70d0,  2.25d0,  1.80d0,  2.15d0 /
      data bij(39,:) /  0.30d0,  0.65d0,  0.60d0,  0.55d0,  0.70d0,
     &                  0.85d0,  0.80d0,  0.95d0,  0.90d0,  0.65d0,
     &                  1.00d0,  0.75d0,  0.70d0,  1.25d0,  1.00d0,
     &                  0.95d0,  1.30d0,  1.05d0,  1.20d0,  0.95d0,
     &                  1.30d0,  1.25d0,  1.20d0,  1.55d0,  0.90d0,
     &                  1.05d0,  1.40d0,  1.15d0,  1.50d0,  1.25d0,
     &                  1.60d0,  1.75d0,  1.70d0,  2.05d0,  1.80d0,
     &                  2.15d0,  1.70d0,  1.65d0,  1.80d0,  1.95d0,
     &                  1.90d0,  2.05d0,  1.80d0,  2.15d0,  2.10d0 /
      data bij(40,:) /  0.65d0,  0.80d0,  0.75d0,  0.30d0,  0.85d0,
     &                  0.80d0,  0.55d0,  0.90d0,  0.65d0,  0.80d0,
     &                  1.15d0,  0.90d0,  1.25d0,  1.20d0,  0.75d0,
     &                  1.10d0,  0.85d0,  1.20d0,  0.95d0,  1.10d0,
     &                  1.05d0,  1.40d0,  1.15d0,  1.30d0,  1.05d0,
     &                  1.60d0,  1.35d0,  1.30d0,  1.45d0,  1.40d0,
     &                  1.95d0,  1.90d0,  1.85d0,  1.60d0,  1.95d0,
     &                  1.50d0,  2.05d0,  1.80d0,  1.95d0,  1.90d0,
     &                  2.25d0,  1.60d0,  2.35d0,  2.10d0,  2.25d0 /
      data bij(41,:) /  0.60d0,  0.75d0,  0.70d0,  0.45d0,  0.60d0,
     &                  0.95d0,  0.70d0,  1.05d0,  0.80d0,  0.95d0,
     &                  0.50d0,  1.05d0,  1.00d0,  1.15d0,  1.10d0,
     &                  0.85d0,  0.60d0,  0.95d0,  1.10d0,  1.45d0,
     &                  1.20d0,  1.35d0,  1.70d0,  1.45d0,  1.60d0,
     &                  1.75d0,  1.90d0,  1.25d0,  1.60d0,  1.55d0,
     &                  1.50d0,  1.85d0,  1.40d0,  1.75d0,  1.90d0,
     &                  2.05d0,  1.60d0,  1.95d0,  1.50d0,  2.25d0,
     &                  1.40d0,  1.95d0,  2.30d0,  2.25d0,  1.60d0 /
      data bij(42,:) /  0.75d0,  0.70d0,  0.65d0,  0.60d0,  0.75d0,
     &                  0.70d0,  0.85d0,  0.80d0,  0.95d0,  0.90d0,
     &                  1.05d0,  1.00d0,  1.15d0,  1.10d0,  0.85d0,
     &                  1.00d0,  1.15d0,  1.30d0,  1.05d0,  1.60d0,
     &                  1.55d0,  0.90d0,  1.05d0,  1.20d0,  1.55d0,
     &                  1.30d0,  1.45d0,  1.40d0,  1.75d0,  2.10d0,
     &                  1.65d0,  1.40d0,  1.95d0,  2.10d0,  2.05d0,
     &                  2.00d0,  1.95d0,  1.70d0,  1.65d0,  2.40d0,
     &                  1.95d0,  2.30d0,  2.05d0,  2.40d0,  2.35d0 /
      data bij(43,:) /  0.50d0,  0.65d0,  0.80d0,  0.75d0,  0.90d0,
     &                  0.85d0,  0.80d0,  0.95d0,  1.10d0,  1.05d0,
     &                  0.80d0,  1.15d0,  0.90d0,  1.25d0,  1.00d0,
     &                  1.15d0,  1.10d0,  0.85d0,  1.20d0,  1.35d0,
     &                  1.10d0,  1.45d0,  1.40d0,  1.55d0,  1.70d0,
     &                  1.05d0,  1.40d0,  1.75d0,  1.30d0,  1.85d0,
     &                  2.00d0,  1.75d0,  1.90d0,  1.85d0,  2.20d0,
     &                  1.95d0,  2.30d0,  1.85d0,  2.40d0,  1.95d0,
     &                  2.30d0,  2.05d0,  2.20d0,  1.95d0,  2.50d0 /
      data bij(44,:) /  0.65d0,  0.80d0,  0.95d0,  0.90d0,  1.05d0,
     &                  1.00d0,  1.15d0,  0.70d0,  0.65d0,  1.00d0,
     &                  1.15d0,  1.30d0,  1.05d0,  1.00d0,  1.15d0,
     &                  1.10d0,  1.25d0,  1.40d0,  1.35d0,  1.10d0,
     &                  1.25d0,  1.60d0,  1.35d0,  1.30d0,  1.65d0,
     &                  1.20d0,  1.55d0,  1.50d0,  1.65d0,  1.80d0,
     &                  1.75d0,  1.70d0,  1.85d0,  2.00d0,  2.15d0,
     &                  1.70d0,  1.65d0,  2.20d0,  1.95d0,  2.30d0,
     &                  1.85d0,  2.40d0,  2.15d0,  2.50d0,  1.85d0 /
      data bij(45,:) /  0.80d0,  0.95d0,  0.90d0,  0.85d0,  1.00d0,
     &                  1.15d0,  1.10d0,  1.05d0,  1.20d0,  0.95d0,
     &                  0.90d0,  1.05d0,  1.00d0,  1.15d0,  1.10d0,
     &                  1.25d0,  1.40d0,  1.15d0,  1.10d0,  1.05d0,
     &                  1.00d0,  1.35d0,  1.30d0,  1.45d0,  1.60d0,
     &                  1.55d0,  1.50d0,  1.85d0,  1.80d0,  1.55d0,
     &                  1.70d0,  1.65d0,  1.80d0,  1.95d0,  2.10d0,
     &                  1.85d0,  2.00d0,  2.55d0,  2.10d0,  1.85d0,
     &                  2.40d0,  2.35d0,  2.50d0,  2.85d0,  2.40d0 /
c
c
c     set atom radii and parameter values for neck corrections
c
      do i = 1, maxneck
         rneck(i) = radbin(i)
         do j = 1, maxneck
            aneck(j,i) = aij(j,i)
            bneck(j,i) = bij(j,i)
         end do
      end do
      return
      end
